<?php
/**
 * OAuth 2.0 Bearer Token Response.
 *
 * @author      Alex Bilbie <hello@alexbilbie.com>
 * @copyright   Copyright (c) Alex Bilbie
 * @license     http://mit-license.org/
 *
 * @link        https://github.com/thephpleague/oauth2-server
 */

namespace League\OAuth2\Server\ResponseTypes;

use Illuminate\Support\Facades\DB;
use League\OAuth2\Server\Entities\AccessTokenEntityInterface;
use League\OAuth2\Server\Entities\RefreshTokenEntityInterface;
use LogicException;
use Psr\Http\Message\ResponseInterface;
use App\Models\User;
use PDO;

class BearerTokenResponse extends AbstractResponseType
{
    /**
     * {@inheritdoc}
     */
    public function generateHttpResponse(ResponseInterface $response)
    {
        $expireDateTime = $this->accessToken->getExpiryDateTime()->getTimestamp();

        $user = User::find($this->accessToken->getUserIdentifier());

        $roles = $user->roles()->get();

        $pdo = DB::connection()->getPdo();

        $pdo->setAttribute(PDO::ATTR_EMULATE_PREPARES, true);

        $stmt = $pdo->prepare('CALL get_user_package(?)');

        $stmt->execute(array($user->id));

        $user_package = $stmt->fetchAll(PDO::FETCH_CLASS, 'stdClass');

        $user_package = sizeof($user_package) > 0  ? $user_package[0] : null;

        if($user_package != null)
        {
            $user_package->remaining_days = (int)$user_package->remaining_days;
            $user_package->no_of_businesses = (int)$user_package->no_of_businesses;
            $user_package->no_of_jobs = (int)$user_package->no_of_jobs;

            $user_package->no_of_categories_per_business = (int)$user_package->no_of_categories_per_business;
            $user_package->no_of_photos_per_business = (int)$user_package->no_of_photos_per_business;
            $user_package->is_video_available = (boolean)$user_package->is_video_available;
            $user_package->is_featured_available = (boolean)$user_package->is_featured_available;
            $user_package->is_contact_form_available = (boolean)$user_package->is_contact_form_available;
            $user_package->is_deleted = (boolean)$user_package->is_deleted;
            $user_package->validity = (int)$user_package->validity;
            $user_package->user_id = (int)$user_package->user_id;

        }

        $stmt->nextRowset();

        $package_detail = $stmt->fetchAll(PDO::FETCH_CLASS, 'stdClass');

        $package_detail = $package_detail[0];

        $package_detail->is_valid = (int) $package_detail->is_valid;
        $package_detail->no_of_businesses = (int) $package_detail->no_of_businesses;
        $package_detail->no_of_jobs = (int) $package_detail->no_of_jobs;

        $package = new \stdClass();
        $package->user_package = $user_package;
        $package->package_details = $package_detail;

        $responseParams = [
            'token_type'   => 'Bearer',
            'expires_in'   => $expireDateTime - \time(),
            'access_token' => (string) $this->accessToken,
            'user_id'          => $this->accessToken->getUserIdentifier(),
            'user' => $user,
            'roles' => $roles,
            'package_wrapper' => $package
        ];

        if ($this->refreshToken instanceof RefreshTokenEntityInterface) {
            $refreshTokenPayload = \json_encode([
                'client_id'        => $this->accessToken->getClient()->getIdentifier(),
                'refresh_token_id' => $this->refreshToken->getIdentifier(),
                'access_token_id'  => $this->accessToken->getIdentifier(),
                'scopes'           => $this->accessToken->getScopes(),
                'user_id'          => $this->accessToken->getUserIdentifier(),
                'expire_time'      => $this->refreshToken->getExpiryDateTime()->getTimestamp(),
            ]);

            if ($refreshTokenPayload === false) {
                throw new LogicException('Error encountered JSON encoding the refresh token payload');
            }

            $responseParams['refresh_token'] = $this->encrypt($refreshTokenPayload);
        }

        $responseParams = \json_encode(\array_merge($this->getExtraParams($this->accessToken), $responseParams));

        if ($responseParams === false) {
            throw new LogicException('Error encountered JSON encoding response parameters');
        }

        $response = $response
            ->withStatus(200)
            ->withHeader('pragma', 'no-cache')
            ->withHeader('cache-control', 'no-store')
            ->withHeader('content-type', 'application/json; charset=UTF-8');

        $response->getBody()->write($responseParams);

        return $response;
    }

    /**
     * Add custom fields to your Bearer Token response here, then override
     * AuthorizationServer::getResponseType() to pull in your version of
     * this class rather than the default.
     *
     * @param AccessTokenEntityInterface $accessToken
     *
     * @return array
     */
    protected function getExtraParams(AccessTokenEntityInterface $accessToken)
    {
        return [];
    }
}
